# Text::CPPTemplate 0.3
# copyright (c) 2000, ETH Zurich
# released under the GNU General Public License

package Text::CPPTemplate;

use strict;

use vars qw($VERSION);
$VERSION = 0.3;

=head1 NAME

Text::CPPTemplate - CPP-Style Templates

=head1 SYNOPSIS

 use Text::CPPTemplate;

 my $templ = new Text::CPPTemplate('/var/web/templates','.html');

 print $templ->template({
 	PAGE => 'index',
	ELEMENT => 'header',
	TITLE => 'Test'
 });

=head1 DESCRIPTION

CPPTemplate is a templating system using a CPP-style (C Pre-Processor) syntax.
CPPTemplate is quite fast so that it can be used in online Applications such
as CGI scripts to separate program the code from the HTML. CPPTemplate is not
HTML specific, so it can be used for other applications. For performance
reasons, the files containing the templates are read only once and are cached
for further use. This is especially handy when working with long running
scripts which use the same template over and over again. Apache mod_perl is
such an environment.

An application can use a large number of templates. They could for example represent
different parts of output generated by the aplication.
Each template can contain variables and CPP style if-then-else structures. 
When the template gets activated, all the variables will get substituted
and the if-then-else structures will get processed.

=head1 FILE NAMES

When you activate a template, you do not specify a file-name, but only
variables. Based on the contents of some special variables, CPPTemplate will try
to load an apropriate template file from disk. It tries to do this using a number of
different file-names. The first one to exist will be used.
A directory where the templates reside and a suffix have to be specified
with the C<new> method. The following list shows which variables cause CPPTemplate to
look for which files:

=over 4

=item *

I<PAGE>B<_>I<ELEMENT> (I<PAGE> and I<ELEMENT> are variables)

=item *

I<ELEMENT>

=item *

I<PAGE>

=item *

C<default> (as is, not a variable)

=back

In the example given in L<SYNOPSIS>, the following files will be
opened in turn until one is found to exist (in the directory F</var/web/templates>):
F<index_header.html>, F<header.html>, F<index.html> and F<default.html>.

=head1 VARIABLE SUBSTITUTION

Variables are marked C<##var##> in the templates. If no variable is found with
that specified name, the ##var## text remains unchanged.

=head1 CPP-STYLE DIRECTIVES

"MiniCPP" directives permit the selection of parts of the template based on
some condition. The language is very very basic, it seems to be good-enough
for most applications. The following directives are supported:

=over 4

=item C<// comment>

The whole line is removed from the output

=item C<#ifdef VAR>

If variable VAR is defined, the following text will be selected.

=item C<#if expr>

If the expression C<expr> evaluates to true (see L<"EXPRESSIONS">), the
following text will be selected. You can use substitutions in the expression
with the syntax '##VAR##'.

=item C<#elif expr>

If the previous C<#if> (or C<#elif>) expression was false, evaluate this
C<expr> and if true select the following text.

=item C<#else>

If the previous C<#if> (or C<#elif>) expression was false, select the following text.

=item C<#endif>

Ends an C<#ifdef> or an C<#if>.

=back

Note that these elements can be nested.

The newlines will be removed unless two consecutive lines without MiniCPP
directives are found. Spaces and tabs will be removed from the beginning and
the end of each line. Use '\ ' (backslash space) to insert spaces at the
beginning or the end of the line.


=head1 EXPRESSIONS

At the moment only the following expressions are supported (don't laugh :-))

=over 4

=item A = B

If A is equal to B (the text), then the expression is true.

=item A ~ B

Match A against the regular expression (perl) B. True if it does match, false
otherwise.

=back

=head1 EXAMPLE

 #if ##ELEMENT## = ruler
 <HR>
 #elif ##ELEMENT## = buttons
   #ifdef ADD_URL
     <A href="##ADD_URL##">Add</A>
   #endif
   #ifdef PREV_URL
     <A href="##PREV_URL##">Prev</A>
   #endif
   #ifdef NEXT_URL
     <A href="##NEXT_URL##">Next</A>
   #endif
   </P>
 #endif


=head1 PER-METHOD DOCUMENTATION

=over 4

=cut

######## MINI CPP #########

sub _process_if
{
	my $state= shift;
	my $expr = shift;

	my $last_state = $state->[$#$state];
	if($expr) {
		push @$state, $last_state;
	}
	else {
		push @$state, 0;
	}
}

sub _substitute
{
	$_ = shift;
	my $vars = shift;
	my $val;
	s/##(\w+)##/$val=$vars->{$1};defined $val?$val:"##$1##"/ge;
	return $_;
}

sub _eval_expr
{
	my $expr = shift;
	my $vars = shift;

	$expr =~ s/^\s+//; $expr =~ s/\s+$//;
	if($expr =~ /^(.+?)\s*=\s*(.*)$/) {
		my $a = _substitute($1, $vars);
		my $b = _substitute($2, $vars);
		#print "<BR>@@@ $a eq $b ?\n";
		return $a eq $b;
	}
	elsif($expr =~ /^(.+?)\s*~\s*(.*)$/) {
		my $a = _substitute($1, $vars);
		my $b = _substitute($2, $vars);
		#print "<BR>@@@ $a ~ $b ?\n";
		return ($a =~ /$b/);
	}
	else {
		return $expr;
	}
}

sub _mini_cpp
{
	my $self = shift;
	my $in = shift;
	my $vars = shift;
	my $out = '';

	my @state = (1);
	my $line;
	my $next_linefeed=0;
	foreach $line (@$in) {
#print "@@@@@@ |".join('',@state)."| $line\n";
		if($line !~ /^#[a-z]/) {        # data
			if($state[$#state]) {
				$out .= "\n" if $next_linefeed;
				$out .= _substitute($line,$vars);
				$next_linefeed=1;
			}
			next;
		}
		$next_linefeed=0;
		if($line =~ /^#endif/) {           # #endif
			if($#state<=0) {
				$out .= "\n!!! SYNTAX ERROR: UNEXPECTED #endif !!!\n";
			}
			else {
				$#state--; # fast pop :-)
			}
			next;
		}
		if($line =~ /^#else/) {            # #else
			if($#state<=0) {
				$out .= "\n!!! SYNTAX ERROR: UNEXPECTED #else !!!\n";
			}
			else {
				my $nstate = pop @state ? 0 : 1;
				push @state, ($state[$#state] and $nstate);
			}
			next;
		}
		if($line =~ /^#elif\s+(.+)$/) { # #elif
			if($#state<=0) {
				$out .= "\n!!! SYNTAX ERROR: UNEXPECTED #elif !!!\n";
			}
			else {
				my $nstate = pop @state ? 0 : 1;
				_process_if(\@state, ($state[$#state] and $nstate) ? _eval_expr($1,$vars) : 0);
			}
			next;
		}
		if($line =~ /^#if\s+(.+?)$/) {  # #if
			_process_if(\@state, $state[$#state] ?  _eval_expr($1,$vars) : 0);
			next;
		}
		if($line =~ /^#ifdef\s+(\w+)/) {   # #ifdef
			my $def = $vars->{$1};
			_process_if(\@state, ((defined $def) and ($def ne '')));
			next;
		}

		$out .= "\n!!! SYNTAX ERROR: UNRECOGNIZED TOKEN !!!\n";
	}

	return $out;
}

####### TEMPLATE #######

sub _slurp_file
{
	my $self = shift;
	my $file = shift;

	my $RS_bak = $/;
	undef $/;
	open(SLURP, "<$file") or do {
		return "\n!!! ERROR: couldn't find $file !!!\n";
	};
	my $data = <SLURP> || '';
	close(SLURP);
	$/=$RS_bak;

	# replace '\ ' to a special unprobable string
	$data =~ s/\\ /<<SpAcE>>/g;

	# trim
	$data =~ s/^[ \t]+//gm;
	$data =~ s/[ \t]+$//gm;

	# replace the forced-space string to a space
	$data =~ s/<<SpAcE>>/ /g;

	# strip comments
	$data =~ s|^//.*||gm;

	my @ldata = split("\n",$data,-1);
	return \@ldata;
}

sub _get_template
{
	my $self = shift;
	my $name = shift;

	my $tdir = $self->{dir};
	my $suff = $self->{suff};

	if(!exists $self->{cache}{$name}) {
		my @names = split /\|/, $name;
		my $ok = 0;
		my $filename;
		foreach (@names) {
			$filename = "$tdir/$_$suff";
			if(-r $filename) {
				$self->{cache}{$name} = $self->_slurp_file($filename);
				$ok = 1;
				last;
			}
		}
		if(!$ok) {
			return ["!!! ERROR: couldn't find a template for $name (t-dir: $tdir) !!!"];
		}
	}

	return $self->{cache}{$name};
}

=item C<new($templates_dir, $suffix)>

Create a new CPPTemplate object. C<$templates_dir> is the directory where the templates
are stored and C<$suffix> is a text to append to the file-names.

=cut

sub new {
	my $proto = shift;
	my $tdir = shift;
	my $suff = shift;

	my $class = ref($proto) || $proto;
	my $self  = {};

	$self->{cache} = {};
	$self->{dir}=$tdir;
	$self->{suff}=$suff;

	bless($self,$class);
	return $self;
}

=item C<template(\%vars)>

Return a processed template. C<\%vars> is a hashref containing the variables used for building
the file-name, for the substitutions and the CPP-style directives.

=cut

sub template($)
{
	my $self = shift;
	my $vars = shift;

	my $name = $vars->{PAGE}.'_'.$vars->{ELEMENT}.'|'.
		$vars->{ELEMENT}.'|'.$vars->{PAGE}.'|default';
	my $templ = $self->_get_template($name);
	my $out   = $self->_mini_cpp($templ, $vars);
	chomp $out;
	return $out;
}

1;

=back

=head1 SEE ALSO

Text::TagTemplate(3)

=head1 COPYRIGHT

Copyright (c) 2000 ETH Zurich, All rights reserved.

=head1 LICENSE

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

=head1 AUTHOR

David Schweikert <dws@ee.ethz.ch>,
Tobi Oetiker <oetiker@ee.ethz.ch>
